; -------------- esp.lisp -----------------------------------------------
;
; Esperanto Lisp parser Grammar and Lexicon
; Jui-Yuan Fred Hsu.  May 1995 
;  
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2, or (at your option)
; any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


; -------------------------------------- loading parsers --------------

(in-package :user)

(load "../allen/loadFunction") 

(loadf "LoadParser")
(loadf "Code/Generator")


(defun p (list) (bu-parse list))

(enableSem)                              ; MUST do this for (sem..) to work


; ---------------------------- loading basic morphme lexicon -----------

(load "../vortaro/lex.for.lisp")


; -------------------------------- declare lexical categories ----------

(setq *lexical-cats* '( self corr root pref suff gsuf 

                        art pron suff-partp pron-d2
                      
                      ) )

; grammatical categories: 
;
;  number, adv, prep, noun, adj, verb, proper, pron-demon
;  pp, vp, pp-de, pp-al

; features:
; 
; agr_p (agreement in person), agr_n (in number), agr_do (in direct object, y/n)
; trans (transitive, y/n), v-info (verb information for adj)
; a-info(adjective information), move (no, from, to) (for preposition)
; v-tense (tense: perfect, etc)  v-form (present, past, )
; near ([y/n], is the "cxi" attached?]
;
; semantics:
; 
; lists: (s-pref -)  or (s-pref (ge -))  or (s-pref (ge (mal -))) 
; 
;   s-comp (compound word), s-pref (prefix), s-suff (suffix), 
;   s-vprt (verb participle suffix), s-gsuf (grammar suffix)
; 
; others:
;
; trans (verb: transitive) [y/n],  v-form (verb: present, past..)
; agr_n, agr_p, agr_do [default = 'n']
; pform [simple/poss], s-def [y/n] (definite [article, etc])

; ---------------------------------------- general definitions -------------

(setq *general-grammar*  
 '(cat 

  ((art (lex la)) -gen-1-> (art-abb) ($apost$))
  ((art (lex la)) -gen-2-> (art-alone))
   
)) ; ----- *general-grammar* 


(setq *general-lexicon*
'(  

  ($comma$  ($comma$))
  ($period$ ($period$))
  ($semi$   ($semis$))
  ($apost$  ($apost$))

  (+ (glue))

  (la (art-alone))
  (l  (art-abb))

  (el ($el$)) 
  (de ($de$))
  (al ($al$))

)) ; --- end *general-lexicon*


; ---------------------- self-stand words --------------------------------

(setq *self-grammar*  
 
 '( (headfeatures
      (number lex)
      (adv    lex s-comp s-pref s-vprt s-suff)
      (prep   lex)
      (proper lex)
    )
    
    ( (number) -self-1-> (head (self (sub s_num))) ) 
    ( (prep)   -self-3-> (head (self (sub s_prep))) )
    ( (proper) -self-4-> (head (self (sub s_proper))) )

    ( (adv (move no))   -self-2-> (head (self (sub s_adv))) )
        
  )       
) ; --- end *self-grammar* 


; ---------------------- correlative words --------------------------------

(setq *corr-grammar*  
 
 '( (headfeatures
      (pron-demon  lex agr_p agr_n agr_do)
      (adj         lex s-comp s-pref s-vprt s-suff)
    )

  ( (pron-demon (near y))
    -cor-pron-3->                                  ; pron-demon += cxi
    (head (pron-d2)) (self (lex cxi)))

  ( (pron-demon (near y))
    -cor-pron-4->                                  ; pron-demon += cxi
    (self (lex cxi)) (head (pron-d2)) )

  ( (pron-demon (near n)) 
    -cor-pron-5->                                ; normal simple pron-demon
    (head (pron-d2))) 
        
  ( (adj (a-info pronoun) (agr_n s) (agr_do n))
         -cor-adj-1->                              ; adj = possesive pron
         (head (pron (pform poss))) )

  )       
) ; --- end *corr-grammar* 


(setq *corr-lexicon*
'(  
  
  (mi  (pron (agr_p 1) (agr_n s) (agr_do n) (pform simp))) ; personal pronouns
  (li  (pron (agr_p 3) (agr_n s) (agr_do n) (pform simp)))
  (sxi (pron (agr_p 3) (agr_n s) (agr_do n) (pform simp)))
  (gxi (pron (agr_p 3) (agr_n s) (agr_do n) (pform simp)))
  (ni  (pron (agr_p 1) (agr_n p) (agr_do n) (pform simp)))
  (ili (pron (agr_p 3) (agr_n p) (agr_do n) (pform simp)))
  (vi  (pron (agr_p 2) (agr_do n) (pform simp)
                       (agr_n (? a_n s p)) )) 

  (tiu (pron-d2 (agr_p 3) (agr_n s) (agr_do n)))            ; demonstr. pron
  
)) ; ----- end *corr-lexicon*


; ---------------------- root words --------------------------------

(setq *root-grammar*  
 
 '( (headfeatures
      (r_cmp    lex sub)
      (r_cmp2   lex sub s-comp)
      (r_pref   lex sub s-comp)
      (r_vpart  lex sub s-comp s-pref)
      (r_suff   lex sub v-info s-comp s-pref s-vprt)
      (r_done   lex sub v-info s-comp s-pref s-vprt s-suff)
      (noun_yet lex agr_p agr_n agr_do s-comp s-pref s-vprt s-suff)
      (noun     lex agr_p agr_n agr_do s-comp s-pref s-vprt s-suff)
      (adj      lex sub agr_n agr_do   s-comp s-pref s-vprt s-suff)
      (adv      lex s-comp s-pref s-vprt s-suff)
      (v_raw    lex s-comp s-pref s-vprt s-suff)
    )
  
  ((r_cmp  (s-comp (?l -)))
           -r-cmp-1->                                ; (need at least one to)  
                                                     ; (jump into this state)
           (root (sub s_noun) (lex ?l)) (glue)                
           (head (root)))
           
  ((r_cmp  (s-comp (?l ?sc)))
           -r-cmp-2->                                ; compund root
           (root (sub s_noun) (lex ?l)) (glue) 
           (head (r_cmp (s-comp ?sc))))
         
  ((r_cmp2) -r-cmp-3-> (head (r_cmp)))
  ((r_cmp2) -r-cmp-4-> (head (root)))
    
  ((r_pref) -r-pref-1-> (head (r_cmp2)))             ; add prefix on left
  ((r_pref  (s-pref (?l ?sp)))
            -r-pref-2-> 
            (pref (lex ?l)) 
            (head (r_pref (s-pref ?sp))))
  
  ((r_vpart (v-info n)) -r_vpart-2-> (head (r_pref)))
  ((r_vpart (v-info ?vi) (s-vprt (?l ?sv)))       
            -r_vpart-1->                            ; add verb participle
            (head (r_pref (s-vprt ?sv)))
            (suff-partp (v-info ?vi) (lex ?l)))
    
  ((r_suff) -r-suff-1-> (head (r_vpart)))            ; add suffix on right
  ((r_suff (s-suff (?l ?ss)))
            -r-suff-2-> 
            (head (r_suff (s-suff ?ss)))
            (suff (lex ?l)))
             
  ((r_done (agr_p 3) (agr_n s) (agr_do n)) 
            -r-done-1->                                ; done with affixes
           (head (r_suff)))
  
  ((noun_yet) -r-noun-1->                              ; becomes a noun_yet
              (head (r_done)) (gsuf (sub s_noun)))

  ((noun)     -r-noun-2->                              ; noun ending with '
              (head (r_done)) ($apost$))

  ((adj (a-info ?info))  
           -r-adj-1->                                ; becomes an adj
           (head (r_done (v-info ?info))) 
           (gsuf (sub s_adj)))

  ((adv (move no))
           -r-adv-1->                                ; becomes an adv
           (head (r_done)) (gsuf (sub s_adv)))

  ((v_raw  (trans y))   
           -r-verb-1->                               ; raw transitive verb
           (head (r_done (sub s_verb_tran))))

  ((v_raw  (trans n))
           -r-verb-2->                               ; raw intrans. verb
           (head (r_done (sub s_verb_intr))))
          
  )       
) ; --- end *root-grammar* 


(setq *root-verb-grammar*  
 
 '( (headfeatures
      (verb    lex trans s-comp s-pref s-vprt s-suff)
    )

  ((verb   (v-form infinite))
           -verb-1->                                 ; inifinite verb
           (head (v_raw)) (gsuf (lex +i)))
  
  ((verb   (v-form present))
           -verb-2->                                 ; present verb
           (head (v_raw)) (gsuf (lex +as)))
  
  ((verb   (v-form past))
           -verb-3->                                 ; past verb
           (head (v_raw)) (gsuf (lex +is)))
  
  ((verb   (v-form future))
           -verb-4->                                 ; future verb
           (head (v_raw)) (gsuf (lex +os)))
  
  ((verb   (v-form cond))
           -verb-5->                                 ; cond verb
           (head (v_raw)) (gsuf (lex +us)))
  
  ((verb   (v-form imper))
           -verb-6->                                 ; imper verb
           (head (v_raw)) (gsuf (lex +u)))
  )       
) ; --- end *root-verb-grammar* 


(setq *root-agree-grammar*  
 
 '( (headfeatures

      (noun_yet   lex agr_p              s-comp s-pref s-vprt s-suff)
      (noun       lex agr_p agr_n agr_do s-comp s-pref s-vprt s-suff)
      (adj        lex sub a-info         s-comp s-pref s-vprt s-suff)
      (adv        lex                    s-comp s-pref s-vprt s-suff) 
      (pron       lex agr_p agr_n ) 
      (pron-d2    lex agr_p )
    )

  ((noun_yet  (agr_n p) (agr_do n) )
              -r-agr-n1->                                 ; noun +j
              (head (noun_yet (agr_n s) (agr_do n))) 
              (gsuf (sub s_morf_j)))

  ((noun_yet  (agr_n ?n) (agr_do y))
              -r-agr-n2->                                 ; noun +n
              (head (noun_yet (agr_n ?n) (agr_do n)))
              (gsuf (sub s_morf_n)))

  ((noun) -r-agr-n3-> (head (noun_yet)))                  ; final form

  ((adj    (agr_n p) (agr_do n))
           -r-agr-adj1->                               ; adj +j
           (head (adj (agr_n s) (agr_do n)))
           (gsuf (sub s_morf_j)))

  ((adj    (agr_n ?n) (agr_do y))
           -r-agr-adj2->                               ; adj +n
           (head (adj (agr_n ?n) (agr_do n)))
           (gsuf (sub s_morf_n)))

  ((adv    (move to))
           -r-agr-adv1->                               ; adv[move to] = adv +n
           (head (adv (move no)))
           (gsuf (sub s_morf_n)))

  ((pron   (agr_do n) (pform poss)) 
           -r-agr-pr2->                                   ; pronoun + a
           (head (pron (agr_do n) (pform simp)))
           (gsuf (sub s_adj)))

  ((pron   (agr_do y) (pform simp))                       ; pronoun + n
           -r-agr-pr1->
           (head (pron (agr_do n) (pform simp)))
           (gsuf (sub s_morf_n)))

  ((pron-d2  (agr_n p) (agr_do n))
              -r-agr-prde1->                             ; pron-d2 +j
             (head (pron-d2 (agr_n s) (agr_do n)))
             (gsuf (sub s_morf_j)))

  ((pron-d2  (agr_n ?n) (agr_do y))                        ; pron-d2 + n
              -r-agr-prde2->
             (head (pron-d2 (agr_n ?n) (agr_do n)))
             (gsuf (sub s_morf_n)))

  )       
) ; --- end *root-agree-grammar* 


(setq *root-lexicon*
'(  
  (+ant (suff-partp (v-info prog)))
  (+int (suff-partp (v-info perf)))
  (+ont (suff-partp (v-info futu)))
  (+at  (suff-partp (v-info prog-pass)))
  (+it  (suff-partp (v-info perf-pass)))
  (+ot  (suff-partp (v-info futu-pass)))
))


; --------------- semantic processing ---------------------------------------

(setq *sem-grammar*  
 
 '( (headfeatures

   (adj-sem         agr_n agr_do a-info sub)
   (adj-sem2        agr_n agr_do sem) 
   (adj-sem3        agr_n agr_do a-info sem)
   (adj-sem4        agr_n a-info sub sem)
   (noun-sem        lex agr_p agr_n agr_do)
   (pron-demon-sem  agr_n agr_do )
   (adv-sem         move )
   (pp-sem          )
   (pp-de-sem       )
   (pp-al-sem       )
   (verb-sem        lex trans v-form)
   (vp-sem          trans voice agr_n)
   (se-sem          trans voice subj DO IO agent )
  )

  ((adj-sem (sem (adj= (a-info ?ai) (lex ?l) (sub ?su) (agr_n ?an) 
                       (agr_do ?ado) (s-comp ?sc) (s-pref ?sp)
                       (s-vprt ?sv) (s-suff ?ss))))
      -sem-adj-1-> 
      (head (adj (a-info ?ai) (lex ?l) (sub ?su) (agr_n ?an) 
                 (agr_do ?ado) (s-comp ?sc) (s-pref ?sp)
                 (s-vprt ?sv) (s-suff ?ss) ) ))

  ((adj-sem2) -sem-adj-2-> (head (adj-sem (a-info ?ai) (sub ?s))))
  ((adj-sem3) -sem-adj-3-> (head (adj-sem (sub ?s))))
  ((adj-sem4) -sem-adj-4-> (head (adj-sem (agr_do ?ado))))

  ((noun-sem (sem (noun= (lex ?l) (agr_p ?ap) (agr_n ?an) (agr_do ?ado)
                         (s-comp ?sc) (s-pref ?sp) (s-vprt ?sv) (s-suff ?ss))))
      -sem-noun-1->
      (head (noun (lex ?l) (agr_p ?ap) (agr_n ?an) (agr_do ?ado)
                  (s-comp ?sc) (s-pref ?sp) (s-vprt ?sv) (s-suff ?ss))))


  ((pron-demon-sem (sem (pron-demon= (lex ?l) (agr_p ?ap) (agr_n ?an) 
                                     (agr_do ?ado) (near ?ne) )))
      -sem-pron-d-1->
      (head (pron-demon (lex ?l) (agr_p ?ap) (agr_n ?an) 
                        (agr_do ?ado) (near ?ne))))
             

  ((adv-sem (sem (adv= (lex ?l) (move ?mv) (s-comp ?sc) (s-pref ?sp)
                       (s-vprt ?sv) (s-suff ?ss))))
       -sem-adv-1-> 
       (head (adv (lex ?l) (move ?mv) (s-comp ?sc) (s-pref ?sp)
                  (s-vprt ?sv) (s-suff ?ss))))

  ((pp-sem (sem (pp= (lex ?l) (move ?m) (info ?s))))
           -sem-pp-1-> (head (pp (lex ?l) (move ?m) (s-np ?s))))
    
  ((pp-de-sem (sem (pp= (lex ?l) (move ?m) (info ?s))))
              -sem-pp-2-> (head (pp-de (lex ?l) (move ?m) (s-np ?s))))

  ((pp-al-sem (sem (pp= (lex ?l) (move ?m) (info ?s))))
              -sem-pp-3-> (head (pp-al (lex ?l) (move ?m) (s-np ?s))))

  ((verb-sem (sem (verb= (lex ?l) (v-form ?vf) (trans ?t) 
                         (s-comp ?sc) (s-pref ?sp) (s-vprt ?sv) (s-suff ?ss))))
             -sem-verb-1->
             (head (verb (lex ?l) (v-form ?vf) (trans ?t) 
                         (s-comp ?sc) (s-pref ?sp) (s-vprt ?sv) (s-suff ?ss))))

  ((vp-sem (sem (vp= (trans ?tr) (v-form ?vf) (v-tense ?vt) (voice ?v)
                     (agr_n ?n) (s-info ?in))))
           -sem-vp-1->
           (head (vp (trans ?tr) (v-form ?vf) (v-tense ?vt) (voice ?v)
                     (agr_n ?n) (s-info ?in))))

  ((se-sem (sem (sent= ?svp (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) 
                            (s-agent ?sa) (s-pp ?spp) 
                            (DO ?do)(IO ?io)(subj ?s)(agent ?a) )))
           -sem-se-1->
           (head (se2- (sem ?svp) (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) 
                       (s-agent ?sa) (s-pp ?spp)
                       (DO ?do)(IO ?io)(subj ?s)(agent ?a) )))
             
  )       
) ; --- end *sem-grammar* 


; ----------------- noun phrases -----------------------------------------

(setq *np-grammar*  
 
 '( (headfeatures
  
   (np-n1    sem agr_n agr_do)
   (np-n2    sem agr_n agr_do)
   (np-n3    sem agr_n agr_do)
   (np-elem  agr_n agr_do)
   (np-prop  )
   (np-of    agr_n agr_do)
   (np-sem   agr_n agr_do)
  )

  ; --------------------------- adj + noun + adj -----------------------------

  ((np-n1  (s-adj (?s -)))
           -np-n2->                                   ; np-n1 = noun + adj
           (head (noun-sem (agr_n ?an) (agr_do ?ado))) 
           (adj-sem2 (agr_n ?an) (agr_do ?ado) (sem ?s)))

  ((np-n1  (s-adj (?s1 ?s2)))
           -np-n3->                                   ; np-n1 = np-n1 + adj
           (head (np-n1 (agr_n ?an) (agr_do ?ado) (s-adj ?s2))) 
           (adj-sem2 (agr_n ?an) (agr_do ?ado) (sem ?s1)))

  ((np-n2  (s-adj (?s -))) 
           -np-n4->                                   ; np-n2 = adj + noun
           (adj-sem2 (agr_n ?an) (agr_do ?ado) (sem ?s))
           (head (noun-sem (agr_n ?an) (agr_do ?ado))))

  ((np-n2  (s-adj (?s1 ?s2)))
           -np-n5->                                   ; np-n2 = adj + np-n2
           (adj-sem2 (agr_n ?an) (agr_do ?ado) (sem ?s1))
           (head (np-n2 (agr_n ?an) (agr_do ?ado) (s-adj ?s2))))

  ((np-n1  (s-adj ?s))
           -np-n6->                                   ; np-n1 = np-n2
           (head (np-n2 (s-adj ?s))))
           
  ((np-n3 (s-adj ?s)) -np-n7-> (head (np-n1 (s-adj ?s))))
  ((np-n3 (s-adj -))  -np-n8-> (head (noun-sem)))

  ; --------------------------- article --------------------------------

  ((np-elem  (sem (noun def ?sadj ?snoun)))
             -np-art1->                                  ; art + noun phrase
             (art) 
             (head (np-n3 (s-adj ?sadj) (sem ?snoun))))  
             
  ((np-elem  (sem (noun indef ?sadj ?snoun)))
             -np-art2-> 
             (head (np-n3 (s-adj ?sadj) (sem ?snoun)))) 

  ; --------------------------- proper names --------------------------------

  ((np-prop  (s-prop (?l -)) (agr_n s) (agr_do n))
             -np-prop1->   
             (head (proper (lex ?l))))

  ((np-prop  (s-prop (?l1 ?l2)) (agr_n ?an) (agr_do ?ado))
             -np-prop2->
             (head (np-prop (s-prop ?l2) (agr_n ?an) (agr_do ?ado))) 
             (proper (lex ?l1)))

  ((np-elem  (sem (proper ?sp)))
             -np-prop4-> (head (np-prop (s-prop ?sp))))

  ; -------------------------- art + possess-pronoun ---------------------
  
  ((np-elem (sem (pron-poss ?s)))
            -np-poss1->
            (art) (head (adj-sem3 (a-info pronoun) (sem ?s))))

  ; ------------------- other pronouns -------------------------------

  ((np-elem (sem (pron-simp ?l (agr_p ?ap) (agr_n ?an) (agr_do ?ado))))
            -np-of3-> 
            (head (pron (pform simp) (lex ?l)
                        (agr_p ?ap) (agr_n ?an) (agr_do ?ado))))
  
  ((np-elem (sem ?s)) 
            -np-of4-> 
            (head (pron-demon-sem (sem ?s))))

  ; --------------------------- of --------------------------------

  ((np-of  (sem (?s -)))
           -np-of1->
           (head (np-elem (sem ?s))))

  ((np-of  (sem (?s1 ?s2)))
           -np-of2->
           (head (np-of (sem ?s2)))
           ($de$)
           (np-elem (sem ?s1)))

  ; --------------------------- components of noun phrase --------------------

  ((np-sem (sem (np= ?s))) -np-comp1-> (head (np-of (sem ?s))))
  
  )       
) ; --- end *np-grammar* 


; -------------------------- prepositional phrase -------------------------

(setq *pp-grammar*  
 
 '( cat

  ((pp (move no) (lex ?l) (s-np ?s)) 
       -pp-1->                                  ; pp = prep + noun[not DO]
       (prep (lex ?l))
       (np-sem (agr_do n) (sem ?s)))

  ((pp (move to) (lex ?l) (s-np ?s))
       -pp-2->                                  ; pp[to] = prep + noun[is DO]
       (prep (lex ?l))
       (np-sem (agr_do y) (sem ?s)))
    
  ((pp (move from) (lex ?l) (s-np ?s))
       -pp-3->                                  ; pp[from] = de + pp[no]
       ($de$)
       (pp (move no) (lex ?l) (s-np ?s)))
    
  ((pp (move from) (lex en) (s-np ?s))
       -pp-4->                                  ; pp[from] = el (=de en)
       ($el$)
       (np-sem (agr_do n) (sem ?s)))
  
  ((pp (move ?m) (lex adverb) (s-np ?s))             
       -pp-5->                                  ; pp = adverb
       (adv-sem (move ?m) (sem ?s)) )

  
  ((pp-de (move no) (lex de) (s-np ?s))
          -pp-6->                               ; pp-de = "de" + noun[not DO]
          ($de$) (np-sem (agr_do n) (sem ?s)))

  ((pp-al (move no) (lex al) (s-np ?s))
          -pp-7->                               ; pp-al = "al" + noun[not DO]
          ($al$) (np-sem (agr_do n) (sem ?s)))
    
  )       
) ; --- end *pp-grammar* 



; ------------------------ verb phrase ------------------------------------

(setq *vp-grammar*  
 
 '( (headfeatures
      (v-adj   sub agr_n sem)
      (v-adj2  voice v-tense agr_n sem)
      (vp      trans)
    )

 ((v-adj (voice active)(v-tense prog)) 
         -v-adj1-> (head (adj-sem4 (a-info prog))))
 ((v-adj (voice active)(v-tense perf)) 
         -v-adj2-> (head (adj-sem4 (a-info perf))))
 ((v-adj (voice active)(v-tense futu)) 
         -v-adj3-> (head (adj-sem4 (a-info futu))))
 ((v-adj (voice pass)(v-tense prog)) 
         -v-adj4-> (head (adj-sem4 (a-info prog-pass))))
 ((v-adj (voice pass)(v-tense perf)) 
         -v-adj5-> (head (adj-sem4 (a-info perf-pass))))
 ((v-adj (voice pass)(v-tense futu)) 
         -v-adj6-> (head (adj-sem4 (a-info futu-pass))))

 ((v-adj2 (trans y)) -v-adj7-> (head (v-adj (sub s_verb_tran))))
 ((v-adj2 (trans n)) -v-adj8-> (head (v-adj (sub s_verb_intr))))

 ((vp  (v-form ?vf)(v-tense simple)(voice active)(agr_n ?n)(s-info ?s))
       -vp-1->                                               ; simple tense
       (head (verb-sem (v-form ?vf) (sem ?s))))

 ((vp (v-form ?vf) (v-tense ?vt) (voice ?v) (agr_n ?n)
      (s-info (?s1 ?s2)))
      -vp-2->                                               ; compound tenses
      (verb-sem (lex est) (v-form ?vf) (sem ?s1))
      (head (v-adj2 (v-tense ?vt) (voice ?v) (agr_n ?n)(sem ?s2))))

 ((vp  (v-form ?vf)(v-tense simple)(voice active)(agr_n ?an)
       (s-info (?s1 ?s2)))
       -vp-3->                                               ; to-be + adj
       (head (verb-sem (lex est) (v-form ?vf) (sem ?s1)))
       (adj-sem (agr_n ?an) (sem ?s2) (a-info n) (sub ?su)))
 

  )       
) ; --- end *vp-grammar* 


; ---------------------- sentence ----------------------------------------

(setq *sentence-grammar*  
 
'((headfeatures
 
   (se1-      trans voice agr_n sem )
   (se2-      trans voice agr_n sem )
   (s         sem)
  )

  ((se1- (DO n) (IO n) (subj n) (agent n)
         (s-sub -) (s-DO -) (s-IO -) (s-agent -) (s-pp -))
         -se-1->                                ; start point. nothing is found
         (head (vp-sem)))

  ((se1- (DO ?do)(IO ?io)(subj y)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-2->                                        ; se1- = subj np + se1-
         (np-sem (agr_do n) (agr_n ?an) (sem ?ss))
         (head (se1- (DO ?do)(IO ?io)(subj n)(agent ?a)(agr_n ?an)
                     (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))))

  ((se1- (DO y)(IO ?io)(subj ?s)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-3->                                        ; se1- = DO np + se1-
         (np-sem (agr_do y) (sem ?sdo))  
         (head (se1- (DO n)(IO ?io)(subj ?s)(agent ?a)
                     (s-sub ?ss) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))))

  ((se1- (DO ?do)(IO y)(subj ?s)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-4->                                         ; se1- = IO pp + se1-
         (pp-al-sem (sem ?sio))
         (head (se1- (DO ?do)(IO n)(subj ?s)(agent ?a)
                     (s-sub ?ss) (s-DO ?sdo) (s-agent ?sa) (s-pp ?spp))))

  ((se1- (DO ?do)(IO ?io)(subj ?s)(agent y) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-5->                                         ; se1- = DE pp + se1-
         (pp-de-sem (sem ?sa))
         (head (se1- (DO ?do)(IO ?io)(subj ?s)(agent n)
                     (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-pp ?spp))))
  
  ((se1- (DO ?do)(IO ?io)(subj ?s)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp (?s1 ?spp)))
         -se-6->                                                  ; general pp
         (pp-sem (sem ?s1))
         (head (se1- (DO ?do)(IO ?io)(subj ?s)(agent ?a)
            (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))))

  ((se2- (DO ?do)(IO ?io)(subj ?s)(agent ?a)
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-7-> (head (se1- (DO ?do)(IO ?io)(subj ?s)(agent ?a)
            (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))))

  ((se2- (DO ?do)(IO ?io)(subj y)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-12->                                             ; se2- += subj np 
         (head (se2- (DO ?do)(IO ?io)(subj n)(agent ?a)(agr_n ?an)
                     (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp)))
         (np-sem (agr_do n) (agr_n ?an) (sem ?ss)))

  ((se2- (DO y)(IO ?io)(subj ?s)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-13->                                               ; se2- += DO np 
         (head (se2- (DO n)(IO ?io)(subj ?s)(agent ?a)
                     (s-sub ?ss) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp)))
         (np-sem (agr_do y) (sem ?sdo)))

  ((se2- (DO ?do)(IO y)(subj ?s)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-14->                                               ; se2- += IO pp
         (head (se2- (DO ?do)(IO n)(subj ?s)(agent ?a)
                     (s-sub ?ss) (s-DO ?sdo) (s-agent ?sa) (s-pp ?spp)))
         (pp-al-sem (sem ?sio)))

  ((se2- (DO ?do)(IO ?io)(subj ?s)(agent y) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-15->                                               ; se2- += DE pp
         (head (se2- (DO ?do)(IO ?io)(subj ?s)(agent n)
                     (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-pp ?spp)))
         (pp-de-sem (sem ?sa)))
  
  ((se2- (DO ?do)(IO ?io)(subj ?s)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp (?s1 ?spp)))
         -se-16->                                                 ; general pp
         (head (se2- (DO ?do)(IO ?io)(subj ?s)(agent ?a)
            (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp)))
         (pp-sem (sem ?s1)))

  ; ---- this is a repetition --- we want subject to be GENERATED first ---
  ((se2- (DO ?do)(IO ?io)(subj y)(agent ?a) 
         (s-sub ?ss) (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))
         -se-32->                                      ; se2- = subj np + se2-
         (np-sem (agr_do n) (agr_n ?an) (sem ?ss))
         (head (se2- (DO ?do)(IO ?io)(subj n)(agent ?a)(agr_n ?an)
                     (s-DO ?sdo) (s-IO ?sio) (s-agent ?sa) (s-pp ?spp))))


  ; ---------------------- check on the integrity -------------------
  
  ((s)
        -sent-2->                                    ; transitive, passive
        (head (se-sem (trans y) (voice pass)
                      (DO ?do)(IO ?io)(subj y)(agent ?a))))

  ((s)
        -sent-3->                                    ; intransitive, active
        (head (se-sem (trans n) (voice active)
                      (DO n)(IO n)(subj y)(agent n))))

  ((s)
        -sent-1->                                    ; transitive, active
        (head (se-sem (trans y) (voice active)
                      (DO y)(IO ?io)(subj y)(agent n))))

  )
) ; --- end *sentence-grammar* 


; ---------------------- CREATE DATABASE --------------------------------

(make-lexicon    *esp-lexicon*)

(augment-lexicon *general-lexicon*)
(make-grammar    *general-grammar*)

(augment-grammar *self-grammar*)

(augment-lexicon *corr-lexicon*)
(augment-grammar *corr-grammar*)

(augment-lexicon *root-lexicon*)
(augment-grammar *root-grammar*)
(augment-grammar *root-verb-grammar*)
(augment-grammar *root-agree-grammar*)

(augment-grammar *sem-grammar*)

(augment-grammar *np-grammar*)
(augment-grammar *pp-grammar*)
(augment-grammar *vp-grammar*)

(augment-grammar *sentence-grammar*)


; ------------------------- switches ---------------------------------


;(set-find-all)
(set-best-first)

(partial-ok)       ; display parsing result even if we do not reach 'sentence'
;(partial-not-ok)

(traceon)
;(verboseon)
;(verboseoff)
;(traceoff)

;(show-chart)
;(show-answers)


; ----------------------------- testing ------------------------------------

(defun ll ()
 (load "esp.lisp")
)

; (p '(cent))                        ; number 
; (p '(ankaux))                      ; adverb
; (p '(en))                          ; preposition

; (p '(la))
; (p '(l $apost$))                   ; l' with apostrophy

; (p '(vi +a))                       ; possesive pronoun (as pron and adj)
;                                    ; as adj: retains a-info as "pronoun"

; (p '(tiu))                         ; demonstrative pronoun (as pron_demon)
; (p '(cxi tiu))                     ; above + cxi (nearness)
; (p '(tiu cxi))                     ; order doesn't matter

; (p '(libr +o))                     ; simple noun 
; (p '(libr $apost$))                ; noun with apostrophy

; (p '(ofic + libr +o))              ; compound noun
; (p '(dom + ofic + libr +o))        ; super compound

; (p '( mal+ libr +o))               ; prefix
; (p '( mal+ ofic + libr +o))
; (p '( re+ mal+ ofic +o))

; (p '( libr +in +o))                ; suffix
; (p '( libr +in +ej +o))  

; (p '( bon +a ))                   ; simple adjective
; (p '( mal+ bon +a))               ; adj with prefix

; (p '( esper +ant +o))             ; verb participle -> noun
; (p '( esper +ant +a))             ; verb partp -> adj (retains a-info)

; (p '( ambaux ))                   ; simple adverb
; (p '( esper +ant +in +e ))        ; adverb

; (p '( skrib +i ))                ; infinite verb (transitive)
; (p '( skrib +as ))               ; present verb

; (p '( pens +as ))                ; infinite, intransitive

; (p '( libr +o +j ))             ; noun: plural
; (p '( libr +o +j +n ))          ; noun, plural, direct object

; (p '( bon +a +j +n ))            ; adj, plural, DO
; (p '( bon +e +n ))               ; adv, move-to with -n ending
; (p '( mi +n ))                   ; pronoun, DO

; (p '( mi +a +j ))               ; mia -> adj, then adj +j (NO person)
; (p '( mi +a +j +n ))            ; mia -> adj, then adj +j +n

; (p '( tiu +j +n ))               ; pron_demon +j +n 
; (p '( cxi tiu +j +n))            ; above + cxi
; (p '( tiu +j +n cxi ))           ; order is not important

; --- noun phrase

; (p '(libr +o +j +n))                        ; simple noun phraase (noun)
; (p '(bon +a libr +o  ))                     ; adj + noun
; (p '(bon +a libr +o bon +a bon +a))         ; adj + noun + adj + adj
; (p '(l $apost$ bon +a libr +o ))            ; art + adj + noun
; (p '(fred hsu))                             ; proper names
; (p '(libr +o de fred ))                     ; np of np
; (p '(mi +n))                                ; personal pronoun
; (p '(la mi +a ))                            ; (3rd person, singular)
; (p '(la mi +a +j))                          ; (3rd person, plural)
; (p '(cxi tiu +j +n))                        ; demonstrative pronoun
; (p '(tiu +j +n cxi))

; (p '(bon +a +j libr +o +j))                 ; match plural
; (p '(bon +a +j +n libr +o +j +n))           ; match D.O.
; (p '(libr +o +j +n bon +a +j +n))       

; ----- verb phrase

; (p '(esper +as ))                           ; simple verb. trans. present

; (p '(esper +ant +a))                        ; adj containing verb particple
; (p '(est +is esper +ant +a))                ; compound: past, act, progres
; (p '(est +as esper +it +a))                 ; pass, present, perfect
; (p '(est +as esper +it +a +j))              ;  plural

; (p '(est +as bon +a))                       ; verb + adj

; -------- prepositional phrases

; (p '(sur tabl +o ))                         ; prep. phrase.  move=no
; (p '(sur la tabl +o +j +n))                 ;  move=to
; (p '(de sur tabl +o ))                      ;  move=from
; (p '(de en  tabl +o ))                      ;  move from
; (p '(el tabl +o))                           ;  move=from, lex = en

; (p '(nord +e ))                             ; adv as prep.phrase. move=no
; (p '(nord +e +n))                           ; move=to 
; (p '(ambaux +n ))                           ; move=to

; ------------ sentence

; (p '(skrib +as))                             ; starting point (vp only)
; (p '(mi skrib +as))                          ; sub + vp
; (p '(mi +n skrib +as))                       ; DO + vp
; (p '(leter +o +n skrib +as))                 ; DO + vp   
; (p '(al mi skrib +as))                       ; IO + vp
; (p '(de mi skrib +as))                       ; agent + vp
; (p '(en libr +o +n skrib +as))               ; pp[to] + vp
; (p '(de en libr +o skrib +as))               ; pp[from] +vp
; (p '(en libr +o sur tabl +o skrib +as))      ; pp + pp +vp

; (p '(skrib +as mi)) 
; (p '(skrib +as mi +n))
; (p '(skrib +as al mi))
; (p '(skrib +as de mi))
; (p '(skrib +as en libr +o +n ))
; (p '(skrib +as en libr +o sur tabl +o))

; (p '(leter +o +n skrib +as mi ))             ; transitive active, subj DO

;                                              ; transitive, active
;                                              ; has OD IO subj pp
;
; (p '(sur tabl +o al ni skrib +as leter +o +n mi))

;                                              ; trans, passive
;                                              ; subj agent(DE) IO pp
;
; (p '(leter +o est +as skrib +it +a de mi al ni sur tabl +o))

;                                              ; intrans, active
;                                              ; subj pp
; (p '(mi est +as fal +ant +a sur tabl +o))

; (p '(ni est +as fal +ant +a +j ))              ; agreement of number


; ----------------------------------

;(p '(la mi +a +j est +is mal+ fal +ant +in +a +j de sur la tabl +o de cxi tiu))


; ---------------- sentence generation / semantics ------------

; (realize (build-gen-constit 'proper '((lex fred))))
; (realize *completed-parse*)

; --- sentence generation from result of last parse
(defun generate()
  (realize *completed-parse*)
)

; ---- sentence generation
; c: category, feat-l: list of features
; the feature list can contain redundant features, such as 'var '1 '2
; (just copy the list from output of parsing on screen

(defun realize2(c feat-l)

   (setq sen-to-real
      (make-constit :cat c
                    :feats feat-l ))

   (save-parse sen-to-real)
   (generate)
)

; (p '(LIBR + OFIC + LIBR))                 ; compound words (2)
; (p '(ge+ libr + ofic ))                   ; prefix (1)
; (p '(ge+ mal+ libr ))                     ; prefix (2)
; (p '(libr +ant ))                         ; verb participle (can only have 1)
; (p '(libr +ej +in))                       ; suffix (2)

; (p '(ge+ libr + ofic +ant +ej +in ))      ; comp, pref, vprt, suff, 

; (p '(libr +ej +o))                            ; complete noun
; (p '(libr +ej $apost$))                       ; a noun too
; (p '(libr +ej +a))                            ; adj
; (p '(libr +ej +e))                            ; adv
; (p '(don +ej ))                               ; transitive
; (p '(fal +ej ))                               ; intransitive

; ------------------------

; (p '(al fred hsu de la lern +ej +o de cornell ge+ ofic + patr +in +o +j 
;  est +as don +int +a +j tiu +n cxi en mi +a mal+ grand +eg +a dom +eg +o ))


; (p '(vi +a +j eks+ ge+ patr +et +o +j est +as don +int +a +j tiu +n cxi 
;      en mi +a mal+ grand +eg +a dom +eg +o ))


;via eksgepatretoj estas donintaj tiun cxi en mia malgrandega domego  



