// --------------------------- Collect.h ------------------------------
// Jui-Yuan Fred Hsu.  May 1995 
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// March 94, Jui-Yuan Fred Hsu
// CollectUnix is based on Collect.h for IBM pc.
// CollectUnix does not use class library, the allocation and maintenance 
//   is done internally. 
// Cells are created by caller, and then ptrs to cells are passed
//   to Collect.  
// Collect assumes ownership of all cells.  It will destroy them
//   when collect itself dies. 
// Collect always allocate contigous space for the whole array
// 'incre' determines the increment by which the array is enlarged
//   whenever resize is requested.  As a result, set incre to big value,
//   when array changes often, so that we reduce the number of times
//   reallocation is needed.
// 'incre': if neg, then we expand twice the size 
// 
// In this version, errors are written to standard error output
// when some data are corrupted (index error, etc), we print err message and
//    exit(-1) program
// When reading error, do not exit, just make stream bad
// 
// Collect is a template class to collect some data type (as array)
// Cell type MUST define:
//  1. copy constructor, = :  used when copying Collect Array
//                              and in CreateCells()
//  2. default constructor :  used before reading a new cell from stream
//  3. << >> :                stream file input/output
//  4. == :                   array comparison 
//
// C's data types, such as int, double, char, etc., can be used as cells
//   although it's a little bit inefficient, since we must 'allocate'
//   memory for each of these pre-defined few-byte variables
//
// (optional) and a special Copy() that returns a newly copied object (ptr)
//   to allow basic data types to be stored in collection too.
// the index is 0-based
// Collect is intended to overcome problems with BDI Array
//   since it doesn't  have ==, =, has no default constr, etc
//    (cannot contain one array within another)
// Collect itself is a >model< object -> it's complete ADT
//
// If use Collect<Pointer-type>  must be careful.
//   Pointers cannot be read in from stream, so C++ does not define >> for it.
//   must redefine >> for each pointer type used.
//
// Note on usage: 
//
//  If a program declares a type Collect<point> 
//  somewhere in the object code there must be at least one
//  var of type Collect type declared.  Otherwise, compiler won't
//  actually compile Collect, and will give linker error such as
//  "can't find Collect<Point> operator =(), etc"
//  This exist, so that even if program that include Collect<point> package
//   is not fully finished, it can still compile and run.
//
// Cell cannot be an abstract class (having function with =0)
//  we won't be able to 'create' objects of Cell with new 
//
// Notes on CC on SGI machines:
// CC's ptlink doesn't work well if we compile programs from AFS directories
// If we are compileing multiple files that include Collect<>, then 
// Try to invoke compiler from local disk (/usr/tmp) 
// cd /usr/tmp
// CC -c -ptr$HOME/collect/ptrepository  $HOME/collect/CollectTest.c++
// CC -ptr$HOME/collect/ptrepository $HOME/collect/CollectTest.o


#ifndef CollectUnix_H
#define CollectUnix_H

#include <stdio.h>
#include <stdlib.h>
#include <fstream.h>                          // C++file stream
#include <iomanip.h>                          // C++ io manipulator
#include <limits.h>                           // system limits 
                              

// -------- debugging tools -- uncomment next line if want to debug
//#define COLLECT_DEBUG


#define COLLECT_INVALID_INDEX  INT_MAX

typedef void* CellReadFunc(istream& is); 


// ========================================= Collect Class definition =====

template <class Cell>
class Collect 
{
  protected:

   int     upper;        // current physical size of array
   int     incre;        // How many cells to increment when realloc needed
   int     shrink;       // Do we want to shrink whenever possible? 
   
   int     size;         // how many cells actually exist
   Cell ** ptr;          // pointer to array of pointers 
 
   void initialize(int up=16, int inc=-1, int sh=0); // Default const

  public:

   Collect(int up=16, int inc=-1, int sh=0);         // Default const
   Collect(Collect& c);                              // copy const
   Collect(istream& is)                              // from stream 
     { initialize(); is >> *this; } 

   virtual ~Collect();                               // make sure to call 
   
   Collect& operator = (Collect& c);                 // = operator
   
   virtual void CreateCells(int n, Cell& tmpl);      // create with template
   
   inline Cell*  operator ()(int index) const;       // return pointer
   inline Cell&  operator [](int index) const;       // return reference

   Cell& First() const { return (*this)[0]; }        // return 1st cell (ref)
   Cell& Last()  const { return (*this)[size-1]; }   // return last cell (ref)

   virtual  Collect* Copy();                         // make a copy

   virtual  Cell* Add(Cell* cell);                   // Add a cell at bottom
   virtual  Cell* AddAt(int i, Cell* cell);          // Add at index location i
   
   virtual  void  DestroyAt(int index);              // Destroy at index
   virtual  int   Destroy(const Cell& cell);         // destroy an object
   virtual  int   Destroy(Cell* cell);               // destroy
   
   virtual  Cell* DetachAt(int index);               // Remove ownership
   virtual  Cell* Detach(const Cell& cell);
   virtual  Cell* Detach(Cell* cell); 
   
   virtual  Cell* ReplaceAt(int i, Cell* cell);      // replace a cell
   
   virtual  void  Interchange(int i1, int i2);       // interchange cells
   
   void         Flush();                             // Destroy all cells
   inline int   Number() const;                      // number of cells

   virtual int Find(const Cell& cell);               // find index of element
   virtual int Find(Cell* cell); 
   
   void     operator << (Collect& c) { MoveLeft(c); }   
   void     operator >> (Collect& c) { MoveRight(c); }  
   Collect& operator += (Collect& c) { return AppendCopy(c); }

   void     MoveLeft   (Collect& c);                  // move cells to left
   void     MoveRight  (Collect& c);                  // move cells to right
   Collect& AppendCopy (Collect& c);                  // append copies to left

   friend int operator == (const Collect<Cell>& c1, const Collect<Cell>& c2);
   friend int operator != (const Collect<Cell>& c1, const Collect<Cell>& c2);

   // for streams in particular, must use 'const' to ensure that
   //  linker can find the function.  this took me half a day
   
   inline friend ostream& operator << (ostream& os, const Collect<Cell>& c)
                 { c.Write(os); return os; } 
   
   inline friend istream& operator >> (istream& is, const Collect<Cell>& c) 
               { ((Collect<Cell>&)c).Read(is);  return is; } 


   istream& CustomizedRead(istream& is, CellReadFunc func)
            { Read(is, func); return is; }

  protected:        // -- g++ bug, virtual delete_cell(int) must appear 
                    //     before void increase(int);
  
   virtual Cell* create_cell_copy(Cell* ce);   
   virtual void  delete_cell(int i) { delete (ptr[i]); } 
   
   virtual void  Write (ostream& os) const; 
   virtual void  Read  (istream& is, CellReadFunc func=NULL); 

  protected:

   static void check_index(int index, int size, char * func_name);
   
   void  increase(int n=0);
   void  decrease();
   
   static void  FileError(istream& is, char *str);
   static void  BugBuster(); 
}; 
 
// =============================== Collect Code ===============================

// ------------------ Collect (as array) (default) --------------------
// Initialize the collection with some max physical size, and increment
// note that the collection is still empty.
// if no value give to upper and incre, this is seen as a default const
//    default for  upper=16, incre=-1
// an incre of 0 defines a fixed-size array, which cannot expand
//    incre of < 0 (negative) -> array expand twice the size 
// an upper of 0 will be changed to 1, to ensure that ptr is always allocated
// sh:  do we want to shrink the array, whenever elements are removed
//      and big spaces are wasted?  
//      (default) False:  the array can only expand, but not shrink,
//                        this is usually good, since arrays will die out
//                        when they go out of scope.
//      true:  Saves memory space, giving up performance (time)
//             should be used carefully.  Each time 'Destroy' or 'Flush'
//             is performed, array will be cut to the minimum sizes possible
// --------------------------------------------------------------------
template <class Cell>  Collect<Cell>::
Collect (int up, int inc, int sh)
{ initialize(up,inc,sh); } 

template <class Cell>  
void Collect<Cell>:: initialize (int up, int inc, int sh)
{
  upper  = up==0? 1:up ;
  incre  = inc;
  size   = 0;                        // initially, no cell exists
  shrink = sh;                       // Shrink? 
  
  ptr = new Cell*[upper];            // allocate physical size
  
  if (!ptr) 
   { 
     fprintf(stderr,"Collect:Constructor: Can't Allocate Memory\n"); 
     exit(-1);
   }
}

// -------------------- Collect (copy constructor) --------------------
// make an identical copy
// see Contructor() and = operator 
// --------------------------------------------------------------------
template <class Cell>  Collect<Cell>::
Collect (Collect<Cell>& c)
{ 
  upper  = c.upper;
  incre  = c.incre;
  shrink = c.shrink;
  size   = 0;                         // initially, no cell exists
  ptr    = NULL;
  
  *this  = c;
}


// --------------------- ~Collect (destructor)  ----------------------
// Destroy all the cells (since collect owns them)
// use function Flush()
// --------------------------------------------------------------------
template <class Cell> Collect<Cell>:: 
~Collect ()
{
  Flush();
  delete ptr;
  ptr = NULL;
}

// ------------------- CreateCells ------------------------------------
// Create n number of new cells in array
// use Add() to add to array
// use the template cell to duplicate
// --------------------------------------------------------------------
template <class Cell>
void Collect<Cell>:: CreateCells(int n, Cell& tmpl)
{
  for (int i=0; i<n; i++)
    Add( create_cell_copy(&tmpl) );
}


// -------------------- =  Collect ------------------------------------
// left Collect is first flushed (destroyed)
// then a copy of right Collect is assigned to left poly
// left array will be readjusted to have the same upper and incre and shrink
// 
// Internally uses Flush() and Add()
// --------------------------------------------------------------------
template <class Cell>
Collect<Cell>&  Collect<Cell>:: operator = (Collect<Cell>& c)
{
  Flush();                        // destroy existing cells
  delete ptr;                     // delete allocated block

  upper  = c.upper;
  incre  = c.incre;
  shrink = c.shrink;
  size   = c.size; 

  ptr = new Cell*[upper];         // allocate physical size
  if (!ptr) 
   { 
     fprintf(stderr, "Collect: = : Cannot Allocate Memory\n"); 
     exit(-1); 
   }

  size = 0; 
  for (int i=0; i<c.size; i++)  Add(c.create_cell_copy(c.ptr[i]));

  return *this;
}

// ------------------- []  reference to cell object -------------------
// given index i, return cell'r reference
// check for invalid index (if index >= size)
// --------------------------------------------------------------------
template <class Cell>
inline Cell& Collect<Cell>:: operator [](int index) const
{
  check_index(index, size, "[]"); 
  return  *(ptr[index]);
}

// ------------------- () pointer to cell object -------------------
// given index i, return cell's ptr
// check for invalid index (if index >= size)
// --------------------------------------------------------------------
template <class Cell>
inline Cell* Collect<Cell>:: operator ()(int index) const
{
  check_index(index, size, "()"); 
  return  ptr[index];
}


// ------------------------- Copy -------------------------------------
// make a copy of this collection and return ptr to the new collect
// note: caller will own this object, and is responsible for destroying it
// --------------------------------------------------------------------
template <class Cell>
Collect<Cell>*  Collect<Cell>:: Copy()
{
  return new Collect<Cell> (*this);
}


// ---------------------Add ------------------------------------------
// Add cell at bottom of array
// Faster than AddAt() 
// If there is not enough space for new cell, increment array by 'incre'
// Once a cell is added to the array, the array assumes ownership of the
//   cell.  In other words, the caller must not 'delete' the cell.
// Collect will destroy the cells when it feels like doing so.
// returns the 'cell' 
// --------------------------------------------------------------------
template <class Cell>
Cell*  Collect<Cell>:: Add(Cell * cell)
{
  increase();
  ptr[size++]=cell;
  return cell;
}

// -----------------------AddAt -----------------------------------------
// Add 'cell' at the index location 'index'
// 'index' can be from 0 (squeezing original 0 entry down), 
//   to array's size (adding after last entry).  Any other value is invalid
// see Add().   This is slower than Add(), since we have to move down cells
// returns the 'cell'
// --------------------------------------------------------------------
template <class Cell>
Cell*  Collect<Cell>:: AddAt(int index, Cell * cell)
{
  check_index(index, size+1, "AddAt()"); 
  
  increase();                                         // see if must increase
  for (int i=size; i>index; i--)  ptr[i]=ptr[i-1];    // move down
  ptr[index] = cell;                                  // store new cell
  size++; 

  return cell;
}


// --------------------- Destroy / DestroyAt -------------------------------
// Destroy one cell at 'index' location.
// Move all other cells up. (takes some time if array is large)
// If 'shrink' is true, and there is enough available space, 
//   physically shrink the array (take some a lot of time).
//   Unless absolute necessary, do not set 'shrink' to True, as every
//   time we Destroy(), the whole array will be re-allocated (shrinked)
// 'index' can be from 0 to (size-1)
//
// ver 1: input index (DestroyAt)
// ver 2: input an object reference    Internally uses Find()
//        return 1 if object deleted, return 0 if object not found
// ver 3: input pointer to an object.  Internally uses Find()
//        return 1 if object deleted, return 0 if object not found
//
// See Detach(), Find()
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: DestroyAt(int index)
{
  check_index(index, size, "DestroyAt()"); 
  
  delete_cell(index);                                 // destroy cell
  
  for (int i=index; i<size-1; i++)  ptr[i]=ptr[i+1];  // move up
  size--;
  decrease();                                         // see if must decrease 
}
 
template <class Cell>
int Collect<Cell>:: Destroy(const Cell& cell)
{
  int i = Find(cell);
  if (i!=COLLECT_INVALID_INDEX) { DestroyAt(i); return 1; }
  else                          { return 0; } 
}

template <class Cell>
int Collect<Cell>:: Destroy(Cell* cell)
{
  int i = Find(cell);
  if (i!=COLLECT_INVALID_INDEX) { DestroyAt(i); return 1; }
  else                          { return 0; } 
}


// --------------------- Detach / DetachAt ---------------------------------
// Detach one cell at 'index' location
// This does not 'delete' the cell.  
//   (ownership is returned to caller)  
//
// Move all other cells up. (takes some time if array is large)
// If 'shrink' is true, and there is enough available space, 
//   physically shrink the array (take some a lot of time).
//   Unless absolute necessary, do not set 'shrink' to True, as every
//   time we Destroy(), the whole array will be re-allocated (shrinked)
// 'index' can be from 0 to (size-1)
// returns the ptr to the detached cell at [index] 
//
// ver 1: input index (index must be correct)
// ver 2: input an object reference    Internally uses Find()
//        on failure (object not found), returns NULL
// ver 3: input pointer to an object.  Internally uses Find()
//        on failure (object not found), returns NULL
//
// See Destroy(), Find()
// --------------------------------------------------------------------
template <class Cell>
Cell*  Collect<Cell>:: DetachAt(int index)
{
  check_index(index, size, "DetachAt()"); 

  Cell * cell_ptr = ptr[index]; 
  
  for (int i=index; i<size-1; i++)  ptr[i]=ptr[i+1];  // move up
  size--;
  decrease();                          // see if must decrease 

  return cell_ptr; 
}

template <class Cell>
Cell* Collect<Cell>:: Detach(const Cell& cell)
{
  int i = Find(cell);
  if (i!=COLLECT_INVALID_INDEX) { return DetachAt(i); }
  else                          { return NULL; }
}

template <class Cell>
Cell* Collect<Cell>:: Detach(Cell* cell)
{
  int i = Find(cell);
  if (i!=COLLECT_INVALID_INDEX) { return DetachAt(i); }
  else                          { return NULL; }
}

// -------------- ReplaceAt -----------------------------------------
// Destroy current cell at location index, and put the new cell in its place
// return the new cell's address
// see AddAt()
// --------------------------------------------------------------------
template <class Cell>
Cell*  Collect<Cell>:: ReplaceAt(int index, Cell* cell)
{
  check_index(index, size, "ReplaceAt()"); 
   
  delete_cell(index);                                // detroy old cell
  ptr[index] = cell;
  return cell;
}
 
// ------------------------- Interchange -------------------------------
// Interchange two cells's content. 
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: Interchange(int i1, int i2)
{
  check_index(i1, size, "Interchange()");
  check_index(i2, size, "Interchange()");
    
  Cell * tmp = ptr[i1];
  
  ptr[i1] = ptr[i2];
  ptr[i2] = tmp; 
}
 
  
// --------------------- Flush ------------------------------------------
// Flush (destroy) all cells.   But still retain upper and incre information
//    and allocated block is still there. 
// Array become of size==0
// compact array if 'shrink' is defined. 
//
// Internally uses Destroy()
// destroy from last element to first, to avoid moving cells
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: Flush()
{
  for (int i=size-1; i>=0; i--)  DestroyAt(i); 
}


// --------------------- Number ---------------------------------------
// return the number of elements in the array
// do not confuse this with the 'upper' size, which indicates the 
//   allocated physical size of the array in memory
// --------------------------------------------------------------------
template <class Cell>
inline int  Collect<Cell>:: Number() const
{  return size; }


// ------------------------ Find --------------------------------
// given an object, find the object's index in the array.
// Return index, if found
// Return COLLECT_INVALID_INDEX, if not found
// 1. Cell* cell version:  compare the value of 'pointers'
// 2. Cell& cell version:  use Cell::operator= to compare elements.
// --------------------------------------------------------------------
template <class Cell>
int Collect<Cell>::Find(const Cell& cell)
{
  for (int i=0; i<size; i++) 
    if (*(ptr[i]) == cell) return i;
  return COLLECT_INVALID_INDEX;
}

template <class Cell>
int Collect<Cell>::Find(Cell* cell)
{
  for (int i=0; i<size; i++) 
    if (ptr[i] == cell) return i;
  return COLLECT_INVALID_INDEX;
}

// ------------- check_index ---------------------------------------
// internal static function.
// check to see if the index is correct.
// Incorrect index:     index<0  or  index>=size
// return if index is correct
// If index is incorrect, printout warning message, and exit(-1) program
// func_name: is a string name identifying the function that called this 
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: check_index(int index, int size, char * func_name)
{
  if (index<0 || index>=size) 
   {
     fprintf(stderr, "Collect:%s: index error, index = %d\n", func_name,index);
     exit(-1);
   }
}

// ------------------------ increase ------------------------------------
// Internal function to increase the physical size of the array.  
// this is done by allocating a new space, and then copying the whole
//   array to the new location.  expensive operation. 
// Note, only 'upper' is increased, 'size' remains the same
//
// n: number of cells to increase.  default is 0
// actually number of cells added is = MAX(n, incre)
//
// If (size!=upper && n==0 [default]) do not increase, simply return.  
//     We still have space
//  -> note: if n is not default, then we will force it to increase
// if (incre==0)    do not increase, we can't increase, error, exit()
// if (incre<0)     we expand twice the current size
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: increase(int n)
{

# ifdef COLLECT_DEBUG
  cout << "increase() enters.  upper==" << upper << "  n=" << n <<endl;
# endif

  if (size!=upper && n==0) return;  // still have space, and we are not forcing
  if (incre==0) 
   {
     fprintf(stderr, "Collect:increase: array cannot be resized\n");
     exit(-1);
   } 
  
  int act_incre = (incre<0? upper: incre);              // expand twice?
  int new_size = upper + (n>act_incre? n: act_incre);   // force more inc? 
   
  Cell** ptrnew = new Cell* [new_size];
  if (!ptrnew) 
   { 
     fprintf(stderr, "Collect:increase: Cannot Allocate Memory\n"); 
     exit(-1); 
   }

  for (int i=0; i<size; i++)  ptrnew[i]=ptr[i];
  delete ptr;
  ptr = ptrnew;
     
  upper=new_size;
}  
  

// -------------------- decrease ----------------------------------------
// internal.  Decrease the array (compact it)
// Remove All remaining available spaces.
// 'upper' will be equal to 'size' after operation.
// works only if 'shrink' is True, otherwise just reutrn
// Allocate a new space (smaller), and then copy existing cells to it.
//   decrease() is a costy operation, and do not use it unless necessary
// if current size ==0 (no cell exists), then allocates 1 cell
//     to keep ptr always non-null
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: decrease()
{
  if (!shrink)     return;
  if (upper==size) return; 

  if (size==0) upper = 1;          // empty array
  else         upper = size;       // adjust new physical size

  Cell** ptrnew = new Cell* [upper];
  if (!ptrnew) 
   { 
     fprintf(stderr, "Collect:derease: Cannot Allocate Memory\n"); 
     exit(-1); 
   }

  for (int i=0; i<size; i++)  ptrnew[i]=ptr[i];
  delete ptr;
  ptr = ptrnew;
}


// ------------------ != ----------------------------------------------
// If any cell is different in the two collections, then return true
// True : size unequal ||  (any cell different) 
// upper and shrink do not matter
// --------------------------------------------------------------------
template <class Cell>
int  operator != (const Collect<Cell>& c1, const Collect<Cell>& c2)
{
  if (c1.size != c2.size) return 1;

  for (int i=0; i<c1.size; i++)
    if (! (c1[i]==c2[i])) return 1;
  return 0;
}

// ---------------------- == ------------------------------------------
// return 1 (true) if all cells are same
// see !=
// --------------------------------------------------------------------
template <class Cell>
int operator == (const Collect<Cell>& c1, const Collect<Cell>& c2)
{
  return ! (c1!=c2);
}

// ------------------------ create_cell_copy() ------------------
// virtual function 
// it's not used as inline function, because
// If a class A wants to include an Collect<A> in its data, 
//  it won't compile if this function is declared as inline function
// In other word, if Cell is an IMCOMPLETE type, then can use inline 
// ---------------------------------------------------------------------
template <class Cell>
Cell* Collect<Cell>:: create_cell_copy(Cell* ce)   
{ 
  return new Cell( *ce );
}

// -------------------- Write (<< Collect) -----------------------------------
// called by << to output this collect array to file stream
// virtual function - derived classes of Collect may use redefine this func
//
// Output collect to file stream
// format:  < # upper incre shrink : cell1 cell2 .... cellLast > 
//   above format is followed by a space
// there is a space between cell elements
// Users can modify the output text file.  #upper incre and shrink can be 
//   optionally ommitted
// see >> for explanation  for 'const'  
// --------------------------------------------------------------------
template <class Cell>
void Collect<Cell>:: Write(ostream& os) const
{
  os << "< # "  << upper << ' ' << incre << ' ' << shrink << " : ";
  for (int i=0; i<size; i++) os << *(ptr[i]) << ' ';
  os << " > \n"; 
}


#define  CheckChar(is,c,what,str)               \
           if (!is)     FileError(is,str);      \
           if (c!=what) FileError(is,str)


// ---------------------- Read   (>> Collect) ------------------------
// called by >> of Collect and CollectStruct
// virtual function - derived classes of Collect may use redefine this func
//
// Input the collect from file stream
// Current Array will be destroyed totally
// inverse of >> collect
// will trim off leading white spaces and extra spaces between elements
//
// Comments can appear right before collection, because
//  collection scans for '<' as initial signiture.
// It will skip everything before '<'
//
// Can optionally ommit '#upper incre shrink :' 
//   in that case, use Default constructor's parameters.
//
// func: void* func(istream& is)
// A customized function to read in individual cells.
// it should CREATE a new cell of Cell type from io stream, and 
//  return it as (void*)
// Default: func==NULL, no customized read.  use default reading
//
// tech:  Bug in g++
// if we do not declare 'const' for 'co2', then linker cannot find the 
//   function body. 
//  so must type cast to a non-const variable
//
// Internally uses Flush() and Add()
// --------------------------------------------------------------------
template <class Cell> 
void  Collect<Cell> :: Read (istream& is, void* func(istream& is) ) 
{
  char  c=0;
  Cell *cell;
  
  Flush();                       // ------- destroy array and its cells
  delete ptr; 
  ptr = NULL;

  while (c!='<' && !(is.eof())) is >> c;
  CheckChar(is,c,'<', "Collect.h: Read(), '<' not found");

  is >> ws;  c = is.peek();
  if (c=='#')                    // array has upper incre and shrink specifier
   {
     int  up, inc, sh;
     is >>c >>ws >>up >>ws >>inc >>ws >>sh >> c; 
     CheckChar(is,c,':', "Collect.h: Read(), ':' not found"); 
     
     this->initialize(up,inc,sh);   // --------- Call constructor with values
   }
  else this->initialize();          // --------- Call constructor with defaults

  if (func==NULL)
    while ((is>>ws, is.peek()!='>') && !(is.eof()))
     {
       cell = new Cell;                               // default read
       is >> (*cell);
       Add(cell);
     }
  else
    while ((is>>ws, is.peek()!='>') && !(is.eof()))   // customized cell read
      Add( (Cell*) func(is) ); 
     
  is >>ws>>c;
  CheckChar(is,c,'>', "Collect.h: Read(), '>' not found");
}


// ------------------ FileError -----------------------------------
// static internal function. 
// Input a file stream 'is', assumes that an error has occurred at 
//   this point in file stream, displays a message with string 'str' and the 
//   current line number at that file stream position
// Send message to standard error
// set file to be BAD.  So if someone checks on stream, they will 
//   realize that it's corrupted
// ----------------------------------------------------------------
template <class Cell>
void Collect<Cell>:: FileError(istream& is, char *str)
{
  long l = is.tellg();        // find the current file position
  is.clear();                 // clear error bit if any
  is.seekg(0);                // go to the beginning
  char c;
  int  line=1;
  long pos=0;
  while (!(is.eof()) && pos<=l )
   {
     c = is.get();
     if (c=='\n') line++;
     pos++;
   }
  fprintf(stderr, "Reading error at line %d, from func %s\n", line, str);

  is.clear(ios::badbit | ios::failbit | is.rdstate());        // set file flag
}


// ------------------------- BugBuster -----------------------------
// static internal function
// prevent linker error:  Collect<_Point>::<< not defined.
//                        Collect<_Polygon>::<< not defined,  etc
// This function is never actually called.
// ------------------------------------------------------------------
template <class Cell>
void Collect<Cell>:: BugBuster()
{
   Collect<Cell> dum;
   cout << dum; 
}


// -------------------- +=  (AppendCopy) ------------------------------------
// Append all cells in right array into left array
// Right array is unmodified. 
// Return left array as reference
// 
// Note: we do not provide + operator.  reason:
//       usually + is used in right expression, and the whole expression 
//       is usually assigned to another array.  + can only return 'an object'
//        and then a copy of whole expression has to be assigned to left var.
//       This is costy, and can be better done with  x=arr1; x+=arr2;
//
// note: have two versions (operator and member function), beacuse 
//       in some platforms linker can't find the operator version... 
//
// internally uses Add()
// also see:  >>(MoveRight)   <<(MoveLeft)  +=(AppendCopy)
// --------------------------------------------------------------------
template <class Cell>
Collect<Cell>&  Collect<Cell>:: AppendCopy (Collect<Cell>& c)
{

# ifdef COLLECT_DEBUG
  cout << "AppendCopy(): enters.    c.size=" << c.size <<endl;
  cout << "   this->upper=" << this->upper <<endl;
# endif

  if (c.size > (upper-size))  increase(c.size);    // increase physical size 
                                                   // to speed up Add()  

  for (int i=0; i<c.size; i++)                     // make copies of cells
    Add(c.create_cell_copy(c.ptr[i]));             // Add() to this array
  
  return *this;
}    


// -------------------- <<  (MoveLeft) ----------------------------------
// 'move' all cells in right array into left array
// Right array becomes empty array
// no return value
// also see:  >>(MoveRight)   <<(MoveLeft)  +=(AppendCopy)
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: MoveLeft (Collect<Cell>& c)
{
  if (c.size==0) return;                  // if right array is empty, returns

  if (c.size > (upper-size))  
     increase(c.size);                    // increase physical size 

  for (int i=0; i<c.size; i++)            // move pointers to left array
     ptr[size+i] = c.ptr[i];              // left array now OWNS those cells
  size+=c.size; 
  
  c.size = 0;                             // make right array empty
  c.decrease();                           // need to shrink?
}    

// -------------------- >> (MoveRight) ----------------------------------
// 'move' all cells in left array into array array
// left array becomes empty array
// no return value
// also see:  >>(MoveRight)   <<(MoveLeft)  +=(AppendCopy)
// --------------------------------------------------------------------
template <class Cell>
void  Collect<Cell>:: MoveRight (Collect<Cell>& c)
{
  c << *this; 
}


// ========================================= CollectRef Class definition =====

template <class Cell>
class CollectRef : public Collect<Cell>
{
  public:

   CollectRef(int up=16, int inc=-1, int sh=0)                // Default const
    : Collect<Cell>(up,inc,sh) {}

   CollectRef(CollectRef& c): Collect<Cell>(c) {}             // copy const

   CollectRef(istream& is): Collect<Cell>() { is >> *this; }  // from stream 
      
   virtual  Collect<Cell>* Copy() { return new CollectRef<Cell> (*this); }

  protected:

   virtual Cell* create_cell_copy(Cell* ce)   { return ce; }   
   virtual void  delete_cell(int)             { } 
   
   virtual void  Read  (istream& is, CellReadFunc func=NULL); 
};

template <class Cell> 
void  CollectRef<Cell> :: Read (istream& , void* func(istream& is) ) 
{ cerr << "CollectRef: Read().  Reading of Reference Cells not implemented\n";
  func=NULL; }


#endif     // CollectUnix_H 

/* --------------------- revision history -------------------------------
 

aug 30, 94    added 'inline' keyword to friend functions such as ==
              some compilers will treat friend functions inside 'class' 
                 as non-inline by default, causing multiple copies of 
                 friend function when linking 
     - fixed bug in Read().  use 'this->Collect() to call construct  

=== ver 2.0 as of today.   tested. 

oct 19, 94   1. added BugBuster() function 
             2. Modified increase() to accept n as parameter
             3. adjust default 'incre' = 16.  2^4 seems better
                upper = 16 too

             4. +=, << and >> operators

oct 24, 94   5. Find() : 2 versions: with ptr and with object
             6. Destroy(): added 2 versions
             7. Detach(): added 2 versions
             
             8. added 'inline' keyword to Number() function 
                 and to [] and () 
                 
             9. added COLLECT_DEBUG statements
             
oct 24, 94: compile with g++ 2.5.8 on SunOS SparcStation
            test with CollectUnixTest.cc

======= ver 3.0 as of today .   tested. 

nov 6, 94:  Added  CustomizedRead() and modified Read()
            
======= ver 3.1 as of today.  tested.   g++ 2.5.8, SunOS, SparcStation

nov 8, 94: 

  - added <Cell>  to some friend function declaration in class definition.
    some compilers require it (CC on SGI), some do not (g++ on Sparc)
    
  - Renamed Destroy(int i) to DestroyAt(int i)  <- destroy using index
    some compilers cannot distinguish between  the two versions of Destroy()
     int <-> Cell&   (if Cell happens to be 'int') 

  - (above), also, the previous testing (Destroy()) may not be correct. 
    Destroy(Cell& ) for Cell=int may actually cause compiler to invoke 
    wrong function. 

  - (above) same for Detach()
  
  - some compilers don't allow direct Constructor call (in Read()), 
    fixed this. 
  
  - added 'inline' for Number(), [] and () in class definition 
  
  compiled under: 
  
  1.   CC ver 3.2 (use CC +v to see), on SGI Indigo, IRIX.   
       must compile with .C extension or .c++
       !! cannot use .cc !!
       
  2.   DCC, the SGI Delta/C++ compiler, compiles C++ source files using
       Delta/C++ style code generation.
        don't know version 
     
  3.   g++ 2.5.8, SunOS, SparcStation

======= ver 3.2 as of today.  tested.

Dec 3, 94: 

  0. made Collect::Read() virtual
  1. added virtual Collect::Write()
  2. changed files names from CollectUnix* to Collect*
  3. added create_cell_copy(), modified += and = operators
  4. added delete_cell(), modified DestroyAt()

  - removed CollectStruct.h and Collect.doc from package.  they are too old

  5. Created a new derived class   CollectRef<>
     
  - comiple under g++, CC (on SGI), and DCC (on SGI)  [like that of ver 3.2]
  
=========== version 3.3 as of today
 
dec 5, 94

 1. modify increase() and data member incre, 
    now by default array will increase to become twice the current size
 
 2. change exit(0) to exit(-1)
 3. change FileErrorExit() to FileError(), and the function does not 
    terminate program, instead, it set file stream to bad
 
 ========== version 3.4 as of today 
 
dec 14, 94

  1. added new function ReplaceAt()
  2. added new check_index() 

 ========== version 3.5 as of today
 
 dec 16, 94
 
   1. make BugBuster a static function
   2. Introduced AppendCopy(), Moveleft(), and Moveright()
       the operator version does not work correctly in SGI (linker can't find
       it) (using DCC) 
 
 dec 17, 94
 
   1. introduced First() and Last(). 
   
 ========= version 3.6 as of today 

Feb 2, 94

 make create_cell_copy() non-inline function.
 so that it can compile ok if Collect<> is used with a imcomplete class

make Number() a constant function 
 
======================

April 21, 95

added function  CreateCells(int n)
internally uses Add() and Cell's default constructor

changed parameter of create_cell_copy() from int to Cell* 
 
 ---------------------------- to do -------------------------------------

use fstools.h to help indenting output 
 use fstools.h, use ENDL

------------------
                 
*/
